//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <unordered_map>

// template <class Key, class T, class Hash = hash<Key>, class Pred = equal_to<Key>,
//           class Alloc = allocator<pair<const Key, T>>>
// class unordered_multimap

// iterator erase(const_iterator first, const_iterator last)

#include <unordered_map>
#include <string>
#include <set>
#include <cassert>
#include <cstddef>

#include "test_macros.h"
#include "../../../check_consecutive.h"
#include "min_allocator.h"

int main(int, char**) {
  {
    typedef std::unordered_multimap<int, std::string> C;
    typedef std::pair<int, std::string> P;
    P a[] = {
        P(1, "one"),
        P(2, "two"),
        P(3, "three"),
        P(4, "four"),
        P(1, "four"),
        P(2, "four"),
    };
    C c(a, a + sizeof(a) / sizeof(a[0]));
    C::const_iterator i = c.find(2);
    C::const_iterator j = std::next(i, 2);
    C::iterator k       = c.erase(i, i);
    assert(k == i);
    assert(c.size() == 6);
    typedef std::pair<C::iterator, C::iterator> Eq;
    Eq eq = c.equal_range(1);
    assert(std::distance(eq.first, eq.second) == 2);
    std::multiset<std::string> s;
    s.insert("one");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(1), c.end(), 1, s);
    eq = c.equal_range(2);
    assert(std::distance(eq.first, eq.second) == 2);
    s.insert("two");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(2), c.end(), 2, s);
    eq = c.equal_range(3);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 3);
    assert(k->second == "three");
    eq = c.equal_range(4);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 4);
    assert(k->second == "four");
    assert(static_cast<std::size_t>(std::distance(c.begin(), c.end())) == c.size());
    assert(static_cast<std::size_t>(std::distance(c.cbegin(), c.cend())) == c.size());

    k = c.erase(i, j);
    assert(c.size() == 4);
    eq = c.equal_range(1);
    assert(std::distance(eq.first, eq.second) == 2);
    s.insert("one");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(1), c.end(), 1, s);
    eq = c.equal_range(3);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 3);
    assert(k->second == "three");
    eq = c.equal_range(4);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 4);
    assert(k->second == "four");
    assert(static_cast<std::size_t>(std::distance(c.begin(), c.end())) == c.size());
    assert(static_cast<std::size_t>(std::distance(c.cbegin(), c.cend())) == c.size());

    k = c.erase(c.cbegin(), c.cend());
    assert(c.size() == 0);
    assert(k == c.end());
  }
  {   // Make sure we're properly destroying the elements when erasing
    { // When erasing part of a bucket
      std::unordered_multimap<int, std::string> map;
      map.insert(std::make_pair(1, "This is a long string to make sure ASan can detect a memory leak"));
      map.insert(std::make_pair(1, "This is another long string to make sure ASan can detect a memory leak"));
      map.erase(++map.begin(), map.end());
    }
    { // When erasing the whole bucket
      std::unordered_multimap<int, std::string> map;
      map.insert(std::make_pair(1, "This is a long string to make sure ASan can detect a memory leak"));
      map.insert(std::make_pair(1, "This is another long string to make sure ASan can detect a memory leak"));
      map.erase(map.begin(), map.end());
    }
  }
  { // Make sure that we're properly updating the bucket list when starting within a bucket
    struct MyHash {
      size_t operator()(size_t v) const { return v; }
    };
    std::unordered_multimap<size_t, size_t, MyHash> map;
    size_t collision_val = 2 + map.bucket_count(); // try to get a bucket collision
    map.rehash(3);
    map.insert(std::pair<size_t, size_t>(1, 1));
    map.insert(std::pair<size_t, size_t>(collision_val, 1));
    map.insert(std::pair<size_t, size_t>(2, 1));
    LIBCPP_ASSERT(map.bucket(2) == map.bucket(collision_val));

    auto erase = map.equal_range(2);
    map.erase(erase.first, erase.second);
    for (const auto& v : map)
      assert(v.first == 1 || v.first == collision_val);
  }
  { // Make sure that we're properly updating the bucket list when we're erasing to the end
    std::unordered_multimap<int, int> m;
    m.insert(std::make_pair(1, 1));
    m.insert(std::make_pair(2, 2));

    {
      auto pair = m.equal_range(1);
      assert(pair.first != pair.second);
      m.erase(pair.first, pair.second);
    }

    {
      auto pair = m.equal_range(2);
      assert(pair.first != pair.second);
      m.erase(pair.first, pair.second);
    }

    m.insert(std::make_pair(3, 3));
    assert(m.size() == 1);
    assert(*m.begin() == std::make_pair(3, 3));
    assert(++m.begin() == m.end());
  }
#if TEST_STD_VER >= 11
  {
    typedef std::unordered_multimap<int,
                                    std::string,
                                    std::hash<int>,
                                    std::equal_to<int>,
                                    min_allocator<std::pair<const int, std::string>>>
        C;
    typedef std::pair<int, std::string> P;
    P a[] = {
        P(1, "one"),
        P(2, "two"),
        P(3, "three"),
        P(4, "four"),
        P(1, "four"),
        P(2, "four"),
    };
    C c(a, a + sizeof(a) / sizeof(a[0]));
    C::const_iterator i = c.find(2);
    C::const_iterator j = std::next(i, 2);
    C::iterator k       = c.erase(i, i);
    assert(k == i);
    assert(c.size() == 6);
    typedef std::pair<C::iterator, C::iterator> Eq;
    Eq eq = c.equal_range(1);
    assert(std::distance(eq.first, eq.second) == 2);
    std::multiset<std::string> s;
    s.insert("one");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(1), c.end(), 1, s);
    eq = c.equal_range(2);
    assert(std::distance(eq.first, eq.second) == 2);
    s.insert("two");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(2), c.end(), 2, s);
    eq = c.equal_range(3);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 3);
    assert(k->second == "three");
    eq = c.equal_range(4);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 4);
    assert(k->second == "four");
    assert(static_cast<std::size_t>(std::distance(c.begin(), c.end())) == c.size());
    assert(static_cast<std::size_t>(std::distance(c.cbegin(), c.cend())) == c.size());

    k = c.erase(i, j);
    assert(c.size() == 4);
    eq = c.equal_range(1);
    assert(std::distance(eq.first, eq.second) == 2);
    s.insert("one");
    s.insert("four");
    CheckConsecutiveKeys<C::const_iterator>(c.find(1), c.end(), 1, s);
    eq = c.equal_range(3);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 3);
    assert(k->second == "three");
    eq = c.equal_range(4);
    assert(std::distance(eq.first, eq.second) == 1);
    k = eq.first;
    assert(k->first == 4);
    assert(k->second == "four");
    assert(static_cast<std::size_t>(std::distance(c.begin(), c.end())) == c.size());
    assert(static_cast<std::size_t>(std::distance(c.cbegin(), c.cend())) == c.size());

    k = c.erase(c.cbegin(), c.cend());
    assert(c.size() == 0);
    assert(k == c.end());
  }
#endif

  return 0;
}
