"""Read the HY-1C COCTS relative spectral responses.

Data from lwk1542@hotmail.com

NB! The two IR bands are NOT included.

See issue: https://github.com/pytroll/pyspectral/issues/61
"""
import logging
import os

import numpy as np

from pyspectral.raw_reader import InstrumentRSR
from pyspectral.utils import INSTRUMENTS
from pyspectral.utils import convert2hdf5 as tohdf5

LOG = logging.getLogger(__name__)

COCTS_BAND_NAMES = ['ch1', 'ch2', 'ch3', 'ch4', 'ch5', 'ch6', 'ch7', 'ch8']


class COCTS_RSR(InstrumentRSR):
    """Container for the HY-1C COCTS RSR data."""

    def __init__(self, bandname, platform_name):
        """Initialize the class."""
        super(COCTS_RSR, self).__init__(bandname, platform_name, COCTS_BAND_NAMES)

        self.instrument = INSTRUMENTS.get(platform_name, 'cocts')

        self._get_options_from_config()
        self._get_bandfilenames()

        LOG.debug("Filenames: %s", str(self.filenames))
        if self.filenames[bandname] and os.path.exists(self.filenames[bandname]):
            self.requested_band_filename = self.filenames[bandname]
            self._load(bandname)

        else:
            LOG.warning("Couldn't find an existing file for this band: %s",
                        str(self.bandname))

        # To be compatible with VIIRS....
        self.filename = self.requested_band_filename

    def _load(self, bandname, scale=0.001):
        """Load the COCTS RSR data for the band requested.

        Wavelength is given in nanometers.
        """
        data = np.genfromtxt(self.requested_band_filename,
                             unpack=True,
                             names=['wavelength', 'ch1', 'ch2', 'ch3', 'ch4', 'ch5', 'ch6', 'ch7', 'ch8'],
                             skip_header=0)

        wavelength = data['wavelength'] * scale
        response = data[bandname]

        self.rsr = {'wavelength': wavelength, 'response': response}


if __name__ == "__main__":
    for platform_name in ["HY-1C", ]:
        tohdf5(COCTS_RSR, platform_name, COCTS_BAND_NAMES)
