/// Represents the potential statuses a post can take.
///
/// More information about this can be found at <https://indieweb.org/post_status>.
#[derive(Debug, Clone, PartialEq, Hash, Eq, Default)]
pub enum Status {
    /// The content is available for general use.
    #[default]
    Published,
    /// The content is not yet ready for general use.
    Drafted,
    /// The content has been considered "deleted".
    Deleted,
    /// The content has expired from general use.
    Expired,
    /// A custom status that posts can have.
    Other(String),
}

impl serde::Serialize for Status {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl<'de> serde::Deserialize<'de> for Status {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        String::deserialize(deserializer)
            .and_then(|status_str| status_str.parse().map_err(serde::de::Error::custom))
    }
}

impl std::fmt::Display for Status {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(match self {
            Status::Published => "published",
            Status::Drafted => "draft",
            Status::Deleted => "deleted",
            Status::Expired => "expired",
            Status::Other(status) => status.as_str(),
        })
    }
}

impl std::str::FromStr for Status {
    type Err = crate::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let status_str = s.trim().trim_matches('"').to_lowercase();
        if status_str.is_empty() {
            // This is the default status according to the extension spec at
            // <https://indieweb.org/Micropub-extensions#Post_Status>
            return Ok(Self::Published);
        }

        match status_str.as_str() {
            "published" => Ok(Self::Published),
            "draft" => Ok(Self::Drafted),
            "deleted" => Ok(Self::Deleted),
            "expired" => Ok(Self::Expired),
            other => Ok(Self::Other(other.to_string())),
        }
    }
}

#[test]
fn post_status() {
    use std::str::FromStr;
    assert_eq!(Some(Status::Drafted), Status::from_str("draft").ok());
    assert_eq!(Some(Status::Published), Status::from_str("published").ok());
    assert_eq!(Some(Status::Published), "published".parse().ok());
    assert_eq!(Some(Status::Published), Status::from_str("PublisHed").ok());
    assert_eq!(Some(Status::Published), "PublisHed".parse().ok());
}
